function stdcell_lwire_crossvac_gui
% STD-CELL L-WIRE GUI — 2 arms + central CROSS-VAC block (full HfO2 + cross vacuum)
% Central block variant: full HfO2 + cross vacuum (arms width = L_vac)
% Controls all geometric parameters. Live 3D preview.
% NOTE: N is fixed to 4 cells per arm.

%% ========= DEFAULT VALUES =========
P = struct( ...
    'L',24, 'W',10, 'H',2, ...                 % SiO2 base
    'origin',[0 0 0], ...                      % [x0 y0 z0]
    'H_Ti_bottom',1, ...                       % Ti bottom (height only)
    'H_gold_bottom',2, ...                     % Au bottom (height only)
    'L_vac',4, 'H_vac',3, ...                  % trench (Lx, Hz) — cross arms width
    'L_Au_top',16, 'W_Au_top',8, 'H_Au_top',4, ... % top gold (Lx, Wy, Hz)
    'T_film',0.5, ...                          % Ti rim thickness
    'CapSide_center',22 ...                    % Au cap side (central block)
);
N_fixed = 4; % DO NOT CHANGE: exactly 4 cells per arm

%% ========= GUI LAYOUT =========
ui = uifigure('Name','STD-CELL L-WIRE (2 arms + CROSS-VAC central block)', ...
    'Color','w','Position',[80 80 1180 700]);

gl = uigridlayout(ui,[2 2]);
gl.RowHeight = {'1x', 46};
gl.ColumnWidth = {400, '1x'};

% --- control panel
ctrl = uipanel(gl,'Title','Parameters','FontWeight','bold');
ctrl.Layout.Row = 1; ctrl.Layout.Column = 1;
cg = uigridlayout(ctrl,[23 2]); % one row less (no N field)
cg.RowHeight = repmat({28},1,23);
cg.ColumnWidth = {200, 180};
r=1;
ef_L  = addField(cg, r, 'L [nm]', P.L, [0 Inf]); r=r+1;
ef_W  = addField(cg, r, 'W [nm]', P.W, [0 Inf]); r=r+1;
ef_H  = addField(cg, r, 'H [nm]', P.H, [0 Inf]); r=r+1;
ef_Hti= addField(cg, r, 'H_{Ti,bot} [nm]', P.H_Ti_bottom, [0 Inf]); r=r+1;
ef_Hau= addField(cg, r, 'H_{Au,bot} [nm]', P.H_gold_bottom,[0 Inf]); r=r+1;
ef_Lv = addField(cg, r, 'L_{vac} [nm]', P.L_vac, [0 Inf]); r=r+1;
ef_Hv = addField(cg, r, 'H_{vac} [nm]', P.H_vac, [0 Inf]); r=r+1;
ef_Lat= addField(cg, r, 'L_{Au,top} [nm]', P.L_Au_top, [0 Inf]); r=r+1;
ef_Wat= addField(cg, r, 'W_{Au,top} [nm]', P.W_Au_top, [0 Inf]); r=r+1;
ef_Hat= addField(cg, r, 'H_{Au,top} [nm]', P.H_Au_top, [0 Inf]); r=r+1;
ef_Tf = addField(cg, r, 'T_{film,Ti} [nm]',P.T_film, [0 Inf]); r=r+1;
ef_Cap= addField(cg, r, 'CapSide_{center} [nm]',P.CapSide_center,[0 Inf]); r=r+1;

% origin
lbl = uilabel(cg,'Text','origin x0 [nm]','HorizontalAlignment','right','FontWeight','bold');
lbl.Layout.Row = r; lbl.Layout.Column = 1;
uix0 = uieditfield(cg,'numeric','Value',P.origin(1),'Limits',[-Inf Inf]);
uix0.Layout.Row = r; uix0.Layout.Column = 2; r=r+1;

lbl = uilabel(cg,'Text','origin y0 [nm]','HorizontalAlignment','right','FontWeight','bold');
lbl.Layout.Row = r; lbl.Layout.Column = 1;
uiy0 = uieditfield(cg,'numeric','Value',P.origin(2),'Limits',[-Inf Inf]);
uiy0.Layout.Row = r; uiy0.Layout.Column = 2; r=r+1;

lbl = uilabel(cg,'Text','origin z0 [nm]','HorizontalAlignment','right','FontWeight','bold');
lbl.Layout.Row = r; lbl.Layout.Column = 1;
uiz0 = uieditfield(cg,'numeric','Value',P.origin(3),'Limits',[-Inf Inf]);
uiz0.Layout.Row = r; uiz0.Layout.Column = 2; r=r+1;

% --- buttons row
btnPanel = uipanel(gl,'BorderType','none'); btnPanel.Layout.Row=2; btnPanel.Layout.Column=[1 2];
bgl = uigridlayout(btnPanel,[1 7]); bgl.ColumnWidth = {'fit','fit','fit','fit','1x','fit','fit'};
uibutton(bgl,'Text','Reset defaults','ButtonPushedFcn',@(~,~)resetDefaults());
uibutton(bgl,'Text','Update drawing','BackgroundColor',[0.1 0.5 0.1],'FontColor','w','ButtonPushedFcn',@(~,~)updateAndDraw());
uibutton(bgl,'Text','Export SDE.txt','BackgroundColor',[0.10 0.35 0.80],'FontColor','w','ButtonPushedFcn',@(~,~)exportSDE());
uibutton(bgl,'Text','Export Sdevice.txt','BackgroundColor',[0.80 0.35 0.10],'FontColor','w','ButtonPushedFcn',@(~,~)exportSDEVICE());
uilabel(bgl,'Text',''); % spacer
uibutton(bgl,'Text','Save PNG','ButtonPushedFcn',@(~,~)savePNG());
uibutton(bgl,'Text','Close','ButtonPushedFcn',@(~,~)close(ui));

% --- preview area
axPanel = uipanel(gl,'Title','L-wire preview (2 arms + CROSS-VAC central block)','FontWeight','bold');
axPanel.Layout.Row=1; axPanel.Layout.Column=2;
ax = uiaxes('Parent',axPanel,'BackgroundColor','w'); ax.Units = 'normalized'; ax.Position = [0 0 1 1];

% auto-update on value change
allEf = [ef_L,ef_W,ef_H,ef_Hti,ef_Hau,ef_Lv,ef_Hv,ef_Lat,ef_Wat,ef_Hat,ef_Tf,ef_Cap,uix0,uiy0,uiz0];
for k=1:numel(allEf)
    allEf(k).ValueChangedFcn = @(~,~)updateAndDraw();
end

% initial draw
updateAndDraw();

%% ===== nested GUI helpers =====
function ef = addField(parent, row, lbltxt, val, lim)
    hLbl = uilabel(parent,'Text',lbltxt,'HorizontalAlignment','right','FontWeight','bold');
    hLbl.Layout.Row = row; hLbl.Layout.Column = 1;
    ef = uieditfield(parent,'numeric','Value',val, ...
        'Limits',lim, 'LowerLimitInclusive','on','UpperLimitInclusive','on', ...
        'ValueDisplayFormat','%.6g');
    ef.Layout.Row = row; ef.Layout.Column = 2;
end

function resetDefaults()
    P = struct('L',24,'W',10,'H',2,'origin',[0 0 0], ...
        'H_Ti_bottom',1,'H_gold_bottom',2, ...
        'L_vac',4,'H_vac',3, ...
        'L_Au_top',16,'W_Au_top',8,'H_Au_top',4, ...
        'T_film',0.5,'CapSide_center',22);
    ef_L.Value=P.L; ef_W.Value=P.W; ef_H.Value=P.H;
    ef_Hti.Value=P.H_Ti_bottom; ef_Hau.Value=P.H_gold_bottom;
    ef_Lv.Value=P.L_vac; ef_Hv.Value=P.H_vac;
    ef_Lat.Value=P.L_Au_top; ef_Wat.Value=P.W_Au_top; ef_Hat.Value=P.H_Au_top;
    ef_Tf.Value=P.T_film; ef_Cap.Value=P.CapSide_center;
    uix0.Value=P.origin(1); uiy0.Value=P.origin(2); uiz0.Value=P.origin(3);
    updateAndDraw();
end

function savePNG()
    [f,p] = uiputfile('stdcell_lwire_crossvac.png','Save as');
    if isequal(f,0), return; end
    exportgraphics(ax, fullfile(p,f), 'Resolution',300);
end

function exportSDE()
    [f,p] = uiputfile('lwire_crossvac_named_contacts_SDE.txt','Export SDE.txt');
    if isequal(f,0), return; end
    fid = fopen(fullfile(p,f),'w');
    if fid<0
        uialert(ui,'Cannot write the file.','Error','Icon','error'); return;
    end
    try
        fprintf(fid,';; =====================================================================\n');
        fprintf(fid,';; SENTARUS SDE — L-WIRE CrossVac (N=4) + Per-cell Named Contacts\n');
        fprintf(fid,';; Units: micrometers (µm)\n');
        fprintf(fid,';; Top contacts defined:\n');
        fprintf(fid,';; north_1..4, east_1..4, and center X.\n');
        fprintf(fid,';; Index: 1 = far end, 4 = near the center.\n');
        fprintf(fid,';; =====================================================================\n');
        fprintf(fid,'(sde:clear)\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; MAIN PARAMETERS (µm)\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(sde:define-parameter "BottomDielectricWidth" 0.024)\n');
        fprintf(fid,'(sde:define-parameter "CellLength"    0.01)\n');
        fprintf(fid,'(sde:define-parameter "Pitch"    0.009)\n');
        fprintf(fid,'(sde:define-parameter "BottomDielectricThickness" 0.002)\n');
        fprintf(fid,'(sde:define-parameter "AdhesionLayerThickness"    0.001)\n');
        fprintf(fid,'(sde:define-parameter "GoldLayerThickness"    0.002)\n');
        fprintf(fid,'(sde:define-parameter "TrenchMiddleWidth"    0.004)\n');
        fprintf(fid,'(sde:define-parameter "TrenchWallHeight"    0.003)\n');
        fprintf(fid,'(sde:define-parameter "TopGoldWidthX"        0.016)\n');
        fprintf(fid,'(sde:define-parameter "TopGoldHeightY"    0.008)\n');
        fprintf(fid,'(sde:define-parameter "TopGoldThicknessZ"    0.004)\n');
        fprintf(fid,'(sde:define-parameter "TitaniumRimThickness"    0.0005)\n');
        fprintf(fid,'(sde:define-parameter "CenterCapSide"        0.022)\n');
        fprintf(fid,'(sde:define-parameter "x0"            0)\n');
        fprintf(fid,'(sde:define-parameter "y0"            0)\n');
        fprintf(fid,'(sde:define-parameter "z0"            0)\n');
        fprintf(fid,'(define Ncells 4)\n');
        fprintf(fid,'(sde:define-parameter "BlockMaterial"    "HfO2")\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; DERIVED & CONSTANTS\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(sde:define-parameter "TrenchWallWidth"\n');
        fprintf(fid,' (/ (- BottomDielectricWidth TrenchMiddleWidth) 2.0))\n');
        fprintf(fid,'(define H_tot (+ BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness TrenchWallHeight TopGoldThicknessZ))\n');
        fprintf(fid,'(define RimPackWidth (+ TopGoldHeightY (* 2.0 TitaniumRimThickness)))\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; HELPERS\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(define (mk-cuboid name mat x y z dx dy dz)\n');
        fprintf(fid,' (sdegeo:create-cuboid (position x y z)\n');
        fprintf(fid,' (position (+ x dx) (+ y dy) (+ z dz))\n');
        fprintf(fid,' mat))\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; CONTACT SETS\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(sdegeo:define-contact-set "BottomContact" 4 (color:rgb 1 0 0) "##")\n');
        fprintf(fid,'(sdegeo:define-contact-set "TopContact_north_1" 4 (color:rgb 0 0 1) "##")\n');
        fprintf(fid,'(sdegeo:define-contact-set "TopContact_north_2" 4 (color:rgb 0 0 1) "##")\n');
        fprintf(fid,'(sdegeo:define-contact-set "TopContact_north_3" 4 (color:rgb 0 0 1) "##")\n');
        fprintf(fid,'(sdegeo:define-contact-set "TopContact_north_4" 4 (color:rgb 0 0 1) "##")\n');
        fprintf(fid,'(sdegeo:define-contact-set "TopContact_east_1" 4 (color:rgb 0 0 1) "##")\n');
        fprintf(fid,'(sdegeo:define-contact-set "TopContact_east_2" 4 (color:rgb 0 0 1) "##")\n');
        fprintf(fid,'(sdegeo:define-contact-set "TopContact_east_3" 4 (color:rgb 0 0 1) "##")\n');
        fprintf(fid,'(sdegeo:define-contact-set "TopContact_east_4" 4 (color:rgb 0 0 1) "##")\n');
        fprintf(fid,'(sdegeo:define-contact-set "TopContact_X" 4 (color:rgb 0 0 1) "##")\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; CELL +Y (with named top contact)\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(define (CellY prefix x y z Wcell isCut cname)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_SiO2") "SiO2" x y z BottomDielectricWidth Wcell BottomDielectricThickness)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiB") "Titanium" x y (+ z BottomDielectricThickness) BottomDielectricWidth Wcell AdhesionLayerThickness)\n');
        fprintf(fid,' (define AUB (mk-cuboid (string-append prefix "_AuB") "Gold" x y (+ z BottomDielectricThickness AdhesionLayerThickness) BottomDielectricWidth Wcell GoldLayerThickness))\n');
        fprintf(fid,' (sdegeo:set-current-contact-set "BottomContact")\n');
        fprintf(fid,' (sdegeo:set-contact AUB "BottomContact")\n');
        fprintf(fid,' (define ztop (+ z BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness))\n');
        fprintf(fid,' (define Lvac (max 0.0 (min TrenchMiddleWidth BottomDielectricWidth)))\n');
        fprintf(fid,' (define Lleft (/ (- BottomDielectricWidth Lvac) 2.0))\n');
        fprintf(fid,' (define Lright (- BottomDielectricWidth (+ Lleft Lvac)))\n');
        fprintf(fid,' (if (> Lleft 0.0) (mk-cuboid (string-append prefix "_HfO2_L") "HfO2" x y ztop Lleft Wcell TrenchWallHeight))\n');
        fprintf(fid,' (if (> Lvac 0.0) (mk-cuboid (string-append prefix "_Vac") "Vacuum" (+ x Lleft) y ztop Lvac Wcell TrenchWallHeight))\n');
        fprintf(fid,' (if (> Lright 0.0) (mk-cuboid (string-append prefix "_HfO2_R") "HfO2" (+ x Lleft Lvac) y ztop Lright Wcell TrenchWallHeight))\n');
        fprintf(fid,' (define zcap (+ ztop TrenchWallHeight))\n');
        fprintf(fid,' (define xcap (+ x (/ (- BottomDielectricWidth TopGoldWidthX) 2.0)))\n');
        fprintf(fid,' (define ycap (if isCut (+ y (- Wcell (+ TopGoldHeightY TitaniumRimThickness))) (+ y (/ (- CellLength TopGoldHeightY) 2.0))))\n');
        fprintf(fid,' (define AUT (mk-cuboid (string-append prefix "_AuT") "Gold" xcap ycap zcap TopGoldWidthX TopGoldHeightY TopGoldThicknessZ))\n');
        fprintf(fid,' (sdegeo:set-current-contact-set cname)\n');
        fprintf(fid,' (sdegeo:set-contact AUT cname)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFxL") "Titanium" (- xcap TitaniumRimThickness) ycap zcap TitaniumRimThickness TopGoldHeightY TopGoldThicknessZ)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFxR") "Titanium" (+ xcap TopGoldWidthX) ycap zcap TitaniumRimThickness TopGoldHeightY TopGoldThicknessZ)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFyF") "Titanium" (- xcap TitaniumRimThickness) (- ycap TitaniumRimThickness) zcap (+ TopGoldWidthX (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFyB") "Titanium" (- xcap TitaniumRimThickness) (+ ycap TopGoldHeightY) zcap (+ TopGoldWidthX (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n');
        fprintf(fid,' (define xin_min (- xcap TitaniumRimThickness))\n');
        fprintf(fid,' (define xin_max (+ xcap TopGoldWidthX TitaniumRimThickness))\n');
        fprintf(fid,' (define yin_min (- ycap TitaniumRimThickness))\n');
        fprintf(fid,' (define yin_max (+ ycap TopGoldHeightY TitaniumRimThickness))\n');
        fprintf(fid,' (define xb_min x) (define xb_max (+ x BottomDielectricWidth))\n');
        fprintf(fid,' (define yb_min y) (define yb_max (+ y Wcell))\n');
        fprintf(fid,' (define Tleft (max 0.0 (- xin_min xb_min)))\n');
        fprintf(fid,' (define Tright (max 0.0 (- xb_max xin_max)))\n');
        fprintf(fid,' (define Tfront (max 0.0 (- yin_min yb_min)))\n');
        fprintf(fid,' (define Tback (max 0.0 (- yb_max yin_max)))\n');
        fprintf(fid,' (define dxcore (max 0.0 (- xin_max xin_min)))\n');
        fprintf(fid,' (if (> Tleft 0.0) (mk-cuboid (string-append prefix "_SiO2FrL") "SiO2" xb_min yb_min zcap Tleft Wcell TopGoldThicknessZ))\n');
        fprintf(fid,' (if (> Tright 0.0) (mk-cuboid (string-append prefix "_SiO2FrR") "SiO2" xin_max yb_min zcap Tright Wcell TopGoldThicknessZ))\n');
        fprintf(fid,' (if (and (> Tfront 0.0) (> dxcore 0.0)) (mk-cuboid (string-append prefix "_SiO2FrF") "SiO2" xin_min yb_min zcap dxcore Tfront TopGoldThicknessZ))\n');
        fprintf(fid,' (if (and (> Tback 0.0) (> dxcore 0.0)) (mk-cuboid (string-append prefix "_SiO2FrB") "SiO2" xin_min yin_max zcap dxcore Tback TopGoldThicknessZ))\n');
        fprintf(fid,')\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; CELL +X (with named top contact)\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(define (CellX prefix x y z WcellX isCut cname)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_SiO2") "SiO2" x y z WcellX BottomDielectricWidth BottomDielectricThickness)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiB") "Titanium" x y (+ z BottomDielectricThickness) WcellX BottomDielectricWidth AdhesionLayerThickness)\n');
        fprintf(fid,' (define AUB (mk-cuboid (string-append prefix "_AuB") "Gold" x y (+ z BottomDielectricThickness AdhesionLayerThickness) WcellX BottomDielectricWidth GoldLayerThickness))\n');
        fprintf(fid,' (sdegeo:set-current-contact-set "BottomContact")\n');
        fprintf(fid,' (sdegeo:set-contact AUB "BottomContact")\n');
        fprintf(fid,' (define ztop (+ z BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness))\n');
        fprintf(fid,' (define Ly_vac (max 0.0 (min TrenchMiddleWidth BottomDielectricWidth)))\n');
        fprintf(fid,' (define Ly_left (/ (- BottomDielectricWidth Ly_vac) 2.0))\n');
        fprintf(fid,' (define Ly_right (- BottomDielectricWidth (+ Ly_left Ly_vac)))\n');
        fprintf(fid,' (if (> Ly_left 0.0) (mk-cuboid (string-append prefix "_HfO2_L") "HfO2" x y ztop WcellX Ly_left TrenchWallHeight))\n');
        fprintf(fid,' (if (> Ly_vac 0.0) (mk-cuboid (string-append prefix "_Vac") "Vacuum" x (+ y Ly_left) ztop WcellX Ly_vac TrenchWallHeight))\n');
        fprintf(fid,' (if (> Ly_right 0.0) (mk-cuboid (string-append prefix "_HfO2_R") "HfO2" x (+ y Ly_left Ly_vac) ztop WcellX Ly_right TrenchWallHeight))\n');
        fprintf(fid,' (define zcap (+ ztop TrenchWallHeight))\n');
        fprintf(fid,' (define xcap (if isCut (+ x (- WcellX (+ TopGoldHeightY TitaniumRimThickness))) (+ x (/ (- CellLength TopGoldHeightY) 2.0))))\n');
        fprintf(fid,' (define ycap (+ y (/ (- BottomDielectricWidth TopGoldWidthX) 2.0)))\n');
        fprintf(fid,' (define AUT (mk-cuboid (string-append prefix "_AuT") "Gold" xcap ycap zcap TopGoldHeightY TopGoldWidthX TopGoldThicknessZ))\n');
        fprintf(fid,' (sdegeo:set-current-contact-set cname)\n');
        fprintf(fid,' (sdegeo:set-contact AUT cname)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFxL") "Titanium" (- xcap TitaniumRimThickness) ycap zcap TitaniumRimThickness TopGoldWidthX TopGoldThicknessZ)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFxR") "Titanium" (+ xcap TopGoldHeightY) ycap zcap TitaniumRimThickness TopGoldWidthX TopGoldThicknessZ)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFyF") "Titanium" (- xcap TitaniumRimThickness) (- ycap TitaniumRimThickness) zcap (+ TopGoldHeightY (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFyB") "Titanium" (- xcap TitaniumRimThickness) (+ ycap TopGoldWidthX) zcap (+ TopGoldHeightY (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n');
        fprintf(fid,' (define xin_min (- xcap TitaniumRimThickness))\n');
        fprintf(fid,' (define xin_max (+ xcap TopGoldHeightY TitaniumRimThickness))\n');
        fprintf(fid,' (define yin_min (- ycap TitaniumRimThickness))\n');
        fprintf(fid,' (define yin_max (+ ycap TopGoldWidthX TitaniumRimThickness))\n');
        fprintf(fid,' (define xb_min x) (define xb_max (+ x WcellX))\n');
        fprintf(fid,' (define yb_min y) (define yb_max (+ y BottomDielectricWidth))\n');
        fprintf(fid,' (define Tleft (max 0.0 (- xin_min xb_min)))\n');
        fprintf(fid,' (define Tright (max 0.0 (- xb_max xin_max)))\n');
        fprintf(fid,' (define Tfront (max 0.0 (- yin_min yb_min)))\n');
        fprintf(fid,' (define Tback (max 0.0 (- yb_max yin_max)))\n');
        fprintf(fid,' (define dxcore (max 0.0 (- xin_max xin_min)))\n');
        fprintf(fid,' (if (> Tleft 0.0) (mk-cuboid (string-append prefix "_SiO2FrL") "SiO2" xb_min yb_min zcap Tleft BottomDielectricWidth TopGoldThicknessZ))\n');
        fprintf(fid,' (if (> Tright 0.0) (mk-cuboid (string-append prefix "_SiO2FrR") "SiO2" xin_max yb_min zcap Tright BottomDielectricWidth TopGoldThicknessZ))\n');
        fprintf(fid,' (if (and (> Tfront 0.0) (> dxcore 0.0)) (mk-cuboid (string-append prefix "_SiO2FrF") "SiO2" xin_min yb_min zcap dxcore Tfront TopGoldThicknessZ))\n');
        fprintf(fid,' (if (and (> Tback 0.0) (> dxcore 0.0)) (mk-cuboid (string-append prefix "_SiO2FrB") "SiO2" xin_min yin_max zcap dxcore Tback TopGoldThicknessZ))\n');
        fprintf(fid,')\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; CENTRAL BLOCK CROSS-VAC (with named top contact)\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(define (CentralBlockCrossVac prefix x y z cname)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_SiO2") "SiO2" x y z BottomDielectricWidth BottomDielectricWidth BottomDielectricThickness)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiB") "Titanium" x y (+ z BottomDielectricThickness) BottomDielectricWidth BottomDielectricWidth AdhesionLayerThickness)\n');
        fprintf(fid,' (define AUB (mk-cuboid (string-append prefix "_AuB") "Gold" x y (+ z BottomDielectricThickness AdhesionLayerThickness) BottomDielectricWidth BottomDielectricWidth GoldLayerThickness))\n');
        fprintf(fid,' (sdegeo:set-current-contact-set "BottomContact")\n');
        fprintf(fid,' (sdegeo:set-contact AUB "BottomContact")\n');
        fprintf(fid,' (define ztop (+ z BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness))\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_HfO2_Mid") BlockMaterial x y ztop BottomDielectricWidth BottomDielectricWidth TrenchWallHeight)\n');
        fprintf(fid,' (define Lx_v (max 0.0 (min TrenchMiddleWidth BottomDielectricWidth)))\n');
        fprintf(fid,' (define x_v (+ x (/ (- BottomDielectricWidth Lx_v) 2.0)))\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_Vac_V") "Vacuum" x_v y ztop Lx_v BottomDielectricWidth TrenchWallHeight)\n');
        fprintf(fid,' (define Wy_h (max 0.0 (min TrenchMiddleWidth BottomDielectricWidth)))\n');
        fprintf(fid,' (define y_h (+ y (/ (- BottomDielectricWidth Wy_h) 2.0)))\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_Vac_H") "Vacuum" x y_h ztop BottomDielectricWidth Wy_h TrenchWallHeight)\n');
        fprintf(fid,' (define zcap (+ ztop TrenchWallHeight))\n');
        fprintf(fid,' (define capSide (max 0.0 (min CenterCapSide BottomDielectricWidth)))\n');
        fprintf(fid,' (define xcap (+ x (/ (- BottomDielectricWidth capSide) 2.0)))\n');
        fprintf(fid,' (define ycap (+ y (/ (- BottomDielectricWidth capSide) 2.0)))\n');
        fprintf(fid,' (define AUT (mk-cuboid (string-append prefix "_AuT") "Gold" xcap ycap zcap capSide capSide TopGoldThicknessZ))\n');
        fprintf(fid,' (sdegeo:set-current-contact-set cname)\n');
        fprintf(fid,' (sdegeo:set-contact AUT cname)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFxL") "Titanium" (- xcap TitaniumRimThickness) ycap zcap TitaniumRimThickness capSide TopGoldThicknessZ)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFxR") "Titanium" (+ xcap capSide) ycap zcap TitaniumRimThickness capSide TopGoldThicknessZ)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFyF") "Titanium" (- xcap TitaniumRimThickness) (- ycap TitaniumRimThickness) zcap (+ capSide (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiFyB") "Titanium" (- xcap TitaniumRimThickness) (+ ycap capSide) zcap (+ capSide (* 2.0 TitaniumRimThickness)) TitaniumRimThickness TopGoldThicknessZ)\n');
        fprintf(fid,' (define xin_min (- xcap TitaniumRimThickness))\n');
        fprintf(fid,' (define xin_max (+ xcap capSide TitaniumRimThickness))\n');
        fprintf(fid,' (define yin_min (- ycap TitaniumRimThickness))\n');
        fprintf(fid,' (define yin_max (+ ycap capSide TitaniumRimThickness))\n');
        fprintf(fid,' (define xb_min x) (define xb_max (+ x BottomDielectricWidth))\n');
        fprintf(fid,' (define yb_min y) (define yb_max (+ y BottomDielectricWidth))\n');
        fprintf(fid,' (define Tleft (max 0.0 (- xin_min xb_min)))\n');
        fprintf(fid,' (define Tright (max 0.0 (- xb_max xin_max)))\n');
        fprintf(fid,' (define Tfront (max 0.0 (- yin_min yb_min)))\n');
        fprintf(fid,' (define Tback (max 0.0 (- yb_max yin_max)))\n');
        fprintf(fid,' (define dxcore (max 0.0 (- xin_max xin_min)))\n');
        fprintf(fid,' (if (> Tleft 0.0) (mk-cuboid (string-append prefix "_SiO2FrL") "SiO2" xb_min yb_min zcap Tleft BottomDielectricWidth TopGoldThicknessZ))\n');
        fprintf(fid,' (if (> Tright 0.0) (mk-cuboid (string-append prefix "_SiO2FrR") "SiO2" xin_max yb_min zcap Tright BottomDielectricWidth TopGoldThicknessZ))\n');
        fprintf(fid,' (if (and (> Tfront 0.0) (> dxcore 0.0)) (mk-cuboid (string-append prefix "_SiO2FrF") "SiO2" xin_min yb_min zcap dxcore Tfront TopGoldThicknessZ))\n');
        fprintf(fid,' (if (and (> Tback 0.0) (> dxcore 0.0)) (mk-cuboid (string-append prefix "_SiO2FrB") "SiO2" xin_min yin_max zcap dxcore Tback TopGoldThicknessZ))\n');
        fprintf(fid,')\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; FILLER\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(define (Filler prefix x y z Dx Dy)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_Base") "SiO2" x y z Dx Dy BottomDielectricThickness)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_TiB") "Titanium" x y (+ z BottomDielectricThickness) Dx Dy AdhesionLayerThickness)\n');
        fprintf(fid,' (define AUB (mk-cuboid (string-append prefix "_AuB") "Gold" x y (+ z BottomDielectricThickness AdhesionLayerThickness) Dx Dy GoldLayerThickness))\n');
        fprintf(fid,' (sdegeo:set-current-contact-set "BottomContact")\n');
        fprintf(fid,' (sdegeo:set-contact AUB "BottomContact")\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_HfO2") BlockMaterial x y (+ z BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness) Dx Dy TrenchWallHeight)\n');
        fprintf(fid,' (mk-cuboid (string-append prefix "_SiO2_Top") "SiO2" x y (+ z BottomDielectricThickness AdhesionLayerThickness GoldLayerThickness TrenchWallHeight) Dx Dy TopGoldThicknessZ)\n');
        fprintf(fid,')\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; BUILD NORTH ARM\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(define y_cursor y0)\n');
        fprintf(fid,'(CellY "north1" x0 y_cursor z0 Pitch #t "TopContact_north_1") (set! y_cursor (+ y_cursor Pitch))\n');
        fprintf(fid,'(CellY "north2" x0 y_cursor z0 Pitch #t "TopContact_north_2") (set! y_cursor (+ y_cursor Pitch))\n');
        fprintf(fid,'(CellY "north3" x0 y_cursor z0 Pitch #t "TopContact_north_3") (set! y_cursor (+ y_cursor Pitch))\n');
        fprintf(fid,'(CellY "north4" x0 y_cursor z0 CellLength #f "TopContact_north_4") (set! y_cursor (+ y_cursor CellLength))\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; CENTRAL BLOCK\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(define yB y_cursor)\n');
        fprintf(fid,'(CentralBlockCrossVac "center" x0 yB z0 "TopContact_X")\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; BUILD EAST ARM\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(define xB2 (+ x0 BottomDielectricWidth))\n');
        fprintf(fid,'(define x_cursor xB2)\n');
        fprintf(fid,'(CellX "east4" x_cursor yB z0 CellLength #f "TopContact_east_4") (set! x_cursor (+ x_cursor CellLength))\n');
        fprintf(fid,'(CellX "east3" x_cursor yB z0 Pitch #t "TopContact_east_3") (set! x_cursor (+ x_cursor Pitch))\n');
        fprintf(fid,'(CellX "east2" x_cursor yB z0 Pitch #t "TopContact_east_2") (set! x_cursor (+ x_cursor Pitch))\n');
        fprintf(fid,'(CellX "east1" x_cursor yB z0 Pitch #t "TopContact_east_1") (set! x_cursor (+ x_cursor Pitch))\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; FILLERS\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(define arm_span (+ CellLength (* (- Ncells 1) Pitch)))\n');
        fprintf(fid,'(define yB2 (+ yB BottomDielectricWidth))\n');
        fprintf(fid,'(Filler "filler_NE" xB2 yB2 z0 arm_span arm_span)\n');
        fprintf(fid,'(Filler "filler_NW" (- x0 arm_span) yB2 z0 arm_span arm_span)\n');
        fprintf(fid,'(Filler "filler_SW" (- x0 arm_span) (- yB arm_span) z0 arm_span arm_span)\n');
        fprintf(fid,'(Filler "filler_SE" xB2 (- yB arm_span) z0 arm_span arm_span)\n');
        fprintf(fid,'(Filler "filler_West" (- x0 arm_span) yB z0 arm_span BottomDielectricWidth)\n');
        fprintf(fid,'(Filler "filler_South" x0 yB2 z0 BottomDielectricWidth arm_span)\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,';; END\n');
        fprintf(fid,';; -------------------------------\n');
        fprintf(fid,'(sde:build-mesh "n@node@")\n');
    catch e
        fclose(fid);
        uialert(ui, getReport(e),'Error','Icon','error');
        return;
    end
    fclose(fid);
end

    function exportSDEVICE()
% Export fixed SDevice content as lwire_squarevac_Sdevice.txt
    [f,p] = uiputfile('lwire_squarevac_Sdevice.txt','Export Sdevice.txt');
    if isequal(f,0), return; end
    fid = fopen(fullfile(p,f),'w');
    if fid<0
        uialert(ui,'Cannot write the file.','Error','Icon','error'); return;
    end
    try
        fprintf(fid,'File {\n');
        fprintf(fid,' Grid = "n1_msh.tdr"\n');
        fprintf(fid,' Plot = "n@node@_clock_des.tdr"\n');
        fprintf(fid,' Current = "n@node@_clock_des.plt"\n');
        fprintf(fid,' Parameter = "sdevice.par"\n');
        fprintf(fid,'}\n\n');
        fprintf(fid,'Electrode {\n');
        fprintf(fid,'*BOTTOMCONTACT\n');
        fprintf(fid,' { Name = "BottomContact" Voltage = 0.0 }\n');
        fprintf(fid,'*INPUTS SEQUENCES TOPCONTACT\n');
        fprintf(fid,' { Name = "TopContact_north_1" Voltage = 0.0 }\n');
        fprintf(fid,' { Name = "TopContact_north_2" Voltage = 0.0 }\n');
        fprintf(fid,' { Name = "TopContact_north_3" Voltage = 0.0 }\n');
        fprintf(fid,' { Name = "TopContact_north_4" Voltage = 0.0 }\n');
        fprintf(fid,'*CENTER CELL TOPCONTACT\n');
        fprintf(fid,' { Name = "TopContact_X" Voltage = 0.0 }\n');
        fprintf(fid,'*OUTPUT TOPCONTACT\n');
        fprintf(fid,' { Name = "TopContact_east_1" Voltage = 0.0 }\n');
        fprintf(fid,' { Name = "TopContact_east_2" Voltage = 0.0 }\n');
        fprintf(fid,' { Name = "TopContact_east_3" Voltage = 0.0 }\n');
        fprintf(fid,' { Name = "TopContact_east_4" Voltage = 0.0 }\n');
        fprintf(fid,'}\n\n');
        fprintf(fid,'# ----------------------------\n# PHYSICS\n# ----------------------------\n');
        fprintf(fid,'Physics (Material="HfO2") {\n CondInsulator\n}\n\n');
        fprintf(fid,'Physics (Material="SiO2") {\n CondInsulator\n}\n\n');
        fprintf(fid,'Thermode {\n');
        fprintf(fid,' { Name = "TopContact_north_1" Temperature = 300 SurfaceResistance = 1e-5 }\n');
        fprintf(fid,' { Name = "TopContact_north_2" Temperature = 300 SurfaceResistance = 1e-5 }\n');
        fprintf(fid,' { Name = "TopContact_north_3" Temperature = 300 SurfaceResistance = 1e-5 }\n');
        fprintf(fid,' { Name = "TopContact_north_4" Temperature = 300 SurfaceResistance = 1e-5 }\n');
        fprintf(fid,' { Name = "TopContact_X" Temperature = 300 SurfaceResistance = 1e-5 }\n');
        fprintf(fid,' { Name = "TopContact_east_1" Temperature = 300 SurfaceResistance = 1e-5 }\n');
        fprintf(fid,' { Name = "TopContact_east_2" Temperature = 300 SurfaceResistance = 1e-5 }\n');
        fprintf(fid,' { Name = "TopContact_east_3" Temperature = 300 SurfaceResistance = 1e-5 }\n');
        fprintf(fid,' { Name = "TopContact_east_4" Temperature = 300 SurfaceResistance = 1e-5 }\n');
        fprintf(fid,' { Name = "BottomContact" Temperature = 300 SurfaceResistance = 1e-5 }\n');
        fprintf(fid,'}\n\n');
        fprintf(fid,'# ----------------------------\n# PLOT\n# ----------------------------\n');
        fprintf(fid,'Plot {\n');
        fprintf(fid,' Potential\n ElectricField\n DielectricConstant\n Temperature\n');
        fprintf(fid,' ConductionCurrentDensity/Vector\n DisplacementCurrentDensity/Vector\n TotalCurrent/Vector\n');
        fprintf(fid,' SpaceCharge\n Potential Doping\n');
        fprintf(fid,' BandGap ElectronAffinity\n');
        fprintf(fid,' ConductionBandEnergy ValenceBandEnergy\n');
        fprintf(fid,'}\n\n');
        fprintf(fid,'Math {\n RelErrControl\n Extrapolate\n}\n\n');
        fprintf(fid,'Solve {\n');
        fprintf(fid,' Coupled (Iterations= 100 LineSearchDamping= 1e-8) {Poisson}\n');
        fprintf(fid,' Coupled{ Poisson Temperature Contact CondInsulator }\n');
        fprintf(fid,' Plot(FilePrefix="n@node@_equilibrium")\n\n');
        fprintf(fid,' # =====================================================================\n');
        fprintf(fid,' # ==== CONFIG (cfg1) ====\n');
        fprintf(fid,' quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4\n\n');
        fprintf(fid,' Goal{ name = "BottomContact" voltage = 0.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_east_1" voltage = 3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_2" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_3" voltage = -3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_4" voltage = 3.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_north_1" voltage = -3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_2" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_3" voltage = 3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_4" voltage = -3.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_X" voltage = 0.0 }\n\n');
        fprintf(fid,' plot { range=(0, 1) intervals= 1}\n');
        fprintf(fid,' ){coupled { Poisson Temperature CondInsulator }}\n\n');
        fprintf(fid,' Plot(FilePrefix="n@node@_cfg1")\n\n');
        fprintf(fid,' # ---- return to 0 V on all top contacts\n');
        fprintf(fid,' quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4\n');
        fprintf(fid,' Goal{ name = "BottomContact" voltage = 0.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_east_1" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_2" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_3" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_4" voltage = 0.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_north_1" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_2" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_3" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_4" voltage = 0.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_X" voltage = 0.0 }\n\n');
        fprintf(fid,' plot { range=(0, 1) intervals= 1}\n');
        fprintf(fid,' ){coupled { Poisson Temperature CondInsulator }}\n\n');
        fprintf(fid,' # =====================================================================\n');
        fprintf(fid,' # ==== CONFIG (cfg2) ====\n');
        fprintf(fid,' quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4\n\n');
        fprintf(fid,' Goal{ name = "BottomContact" voltage = 0.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_east_1" voltage = -3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_2" voltage = 3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_3" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_4" voltage = -3.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_north_1" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_2" voltage = 3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_3" voltage = -3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_4" voltage = 0.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_X" voltage = 3.0 }\n\n');
        fprintf(fid,' plot { range=(0, 1) intervals= 1}\n');
        fprintf(fid,' ){coupled { Poisson Temperature CondInsulator }}\n\n');
        fprintf(fid,' Plot(FilePrefix="n@node@_cfg2")\n\n');
        fprintf(fid,' # ---- return to 0 V on all top contacts\n');
        fprintf(fid,' quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4\n');
        fprintf(fid,' Goal{ name = "BottomContact" voltage = 0.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_east_1" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_2" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_3" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_4" voltage = 0.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_north_1" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_2" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_3" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_4" voltage = 0.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_X" voltage = 0.0 }\n\n');
        fprintf(fid,' plot { range=(0, 1) intervals= 1}\n');
        fprintf(fid,' ){coupled { Poisson Temperature CondInsulator }}\n\n');
        fprintf(fid,' # =====================================================================\n');
        fprintf(fid,' # ==== CONFIG (cfg3) ====\n');
        fprintf(fid,' quasistationary (InitialStep = 0.01 Increment = 1.5 MaxStep = 0.05 MinStep=1e-4\n\n');
        fprintf(fid,' Goal{ name = "BottomContact" voltage = 0.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_east_1" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_2" voltage = -3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_3" voltage = 3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_east_4" voltage = 0.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_north_1" voltage = 3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_2" voltage = -3.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_3" voltage = 0.0 }\n');
        fprintf(fid,' Goal{ name = "TopContact_north_4" voltage = 3.0 }\n\n');
        fprintf(fid,' Goal{ name = "TopContact_X" voltage = -3.0 }\n\n');
        fprintf(fid,' plot { range=(0, 1) intervals= 1}\n');
        fprintf(fid,' ){coupled { Poisson Temperature CondInsulator }}\n\n');
        fprintf(fid,' Plot(FilePrefix="n@node@_cfg3")\n\n');
        fprintf(fid,'}\n\n');

        fclose(fid);
        uialert(ui,'SDevice file exported successfully.','Success','Icon','success');
    catch ME
        fclose(fid);
        uialert(ui,sprintf('Error writing SDevice file: %s', ME.message),'Error','Icon','error');
    end
end
function updateAndDraw()
    P.L = ef_L.Value; P.W = ef_W.Value; P.H = ef_H.Value;
    P.H_Ti_bottom = ef_Hti.Value; P.H_gold_bottom = ef_Hau.Value;
    P.L_vac = ef_Lv.Value; P.H_vac = ef_Hv.Value;
    P.L_Au_top = ef_Lat.Value; P.W_Au_top = ef_Wat.Value; P.H_Au_top = ef_Hat.Value;
    P.T_film = ef_Tf.Value; P.CapSide_center = ef_Cap.Value;
    P.origin = [uix0.Value, uiy0.Value, uiz0.Value];
    x0 = P.origin(1); y0 = P.origin(2); z0 = P.origin(3);
    pitch = calcPitchCutY(P);
    N = N_fixed;

    % Total height
    H_tot = P.H + P.H_Ti_bottom + P.H_gold_bottom + P.H_vac + P.H_Au_top;

    % Clear and hold
    cla(ax, 'reset'); hold(ax,'on'); legend(ax,'off');

    % Build geometries
    y_cursor = y0;
    GN1 = buildCellVerts(P, [x0 y_cursor z0], true,  pitch); y_cursor = y_cursor + pitch;
    GN2 = buildCellVerts(P, [x0 y_cursor z0], true,  pitch); y_cursor = y_cursor + pitch;
    GN3 = buildCellVerts(P, [x0 y_cursor z0], true,  pitch); y_cursor = y_cursor + pitch;
    GN4 = buildCellVerts(P, [x0 y_cursor z0], false, P.W);   y_cursor = y_cursor + P.W;

    yB = y_cursor;

    % CENTRAL CROSS-VAC block
    Gc  = buildCentralCrossVacCellVerts(P, [x0 yB z0]);

    xB2 = x0 + P.L;
    x_cursor = xB2;
    GE4 = buildCellXVerts(P, [x_cursor yB z0], false, P.W, P.L); x_cursor = x_cursor + P.W;
    GE3 = buildCellXVerts(P, [x_cursor yB z0], true,  pitch, P.L); x_cursor = x_cursor + pitch;
    GE2 = buildCellXVerts(P, [x_cursor yB z0], true,  pitch, P.L); x_cursor = x_cursor + pitch;
    GE1 = buildCellXVerts(P, [x_cursor yB z0], true,  pitch, P.L); x_cursor = x_cursor + pitch;

    % Fillers (arm_span generalized with N)
    arm_span = P.W + (N-1) * pitch;
    yB2 = yB + P.L;
    FF_NE = buildFillerLayers([xB2 yB2 z0], arm_span, arm_span, [P.H + P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top]);
    FF_NW = buildFillerLayers([x0 - arm_span yB2 z0], arm_span, arm_span, [P.H + P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top]);
    FF_SW = buildFillerLayers([x0 - arm_span yB - arm_span z0], arm_span, arm_span, [P.H + P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top]);
    FF_SE = buildFillerLayers([xB2 yB - arm_span z0], arm_span, arm_span, [P.H + P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top]);
    FX    = buildFillerLayers([x0 - arm_span yB z0], arm_span, P.L, [P.H + P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top]);
    FS    = buildFillerLayers([x0 yB + P.L z0], P.L, arm_span, [P.H + P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top]);

    % Colors and style
    [c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeColor,lineWidth] = colorsAndStyle();
    F = faces6();

    % Draw all
    drawGeom(ax, GN1, c_base, c_ti, c_au, c_vac, a_vac, c_hfo2, edgeColor, lineWidth, F, true);
    drawGeom(ax, GN2, c_base, c_ti, c_au, c_vac, a_vac, c_hfo2, edgeColor, lineWidth, F, false);
    drawGeom(ax, GN3, c_base, c_ti, c_au, c_vac, a_vac, c_hfo2, edgeColor, lineWidth, F, false);
    drawGeom(ax, GN4, c_base, c_ti, c_au, c_vac, a_vac, c_hfo2, edgeColor, lineWidth, F, false);

    drawGeom(ax, Gc,  c_base, c_ti, c_au, c_vac, a_vac, c_hfo2, edgeColor, lineWidth, F, false);

    drawGeom(ax, GE4, c_base, c_ti, c_au, c_vac, a_vac, c_hfo2, edgeColor, lineWidth, F, false);
    drawGeom(ax, GE3, c_base, c_ti, c_au, c_vac, a_vac, c_hfo2, edgeColor, lineWidth, F, false);
    drawGeom(ax, GE2, c_base, c_ti, c_au, c_vac, a_vac, c_hfo2, edgeColor, lineWidth, F, false);
    drawGeom(ax, GE1, c_base, c_ti, c_au, c_vac, a_vac, c_hfo2, edgeColor, lineWidth, F, false);

    drawFillerLayers(ax, FF_NE, c_base, c_au, c_hfo2, c_base, edgeColor, lineWidth, F, 'Filler', true);
    drawFillerLayers(ax, FF_NW, c_base, c_au, c_hfo2, c_base, edgeColor, lineWidth, F, '', false);
    drawFillerLayers(ax, FF_SW, c_base, c_au, c_hfo2, c_base, edgeColor, lineWidth, F, '', false);
    drawFillerLayers(ax, FF_SE, c_base, c_au, c_hfo2, c_base, edgeColor, lineWidth, F, '', false);
    drawFillerLayers(ax, FX,    c_base, c_au, c_hfo2, c_base, edgeColor, lineWidth, F, '', false);
    drawFillerLayers(ax, FS,    c_base, c_au, c_hfo2, c_base, edgeColor, lineWidth, F, '', false);

    % --- axes & view
    axis(ax,'equal'); grid(ax,'off'); view(ax,35,25);
    xlabel(ax,'X [nm]'); ylabel(ax,'Y [nm]'); zlabel(ax,'Z [nm]');
    title(ax, sprintf('L-wire (2 arms + CROSS-VAC block) — N=%d, pitch=%.3g nm', N, pitch));

    % Spans generalized with N
    arm_span = P.W + (N-1) * pitch;
    Y_span = arm_span + P.L + arm_span;    % north arm + center + south filler
    X_span = P.L + arm_span;               % center + east arm
    marg = 0.12 * [X_span, Y_span, H_tot];

    xlim(ax,[x0 - arm_span - marg(1), x0 + P.L + arm_span + marg(1)]);
    ylim(ax,[y0 - arm_span - marg(2), y0 + Y_span + marg(2)]);
    zlim(ax,[z0 - marg(3), z0 + H_tot + marg(3)]);

    plot3(ax,P.origin(1),P.origin(2),P.origin(3),'ko','MarkerFaceColor','k','HandleVisibility','off');
    legend(ax,'Location','bestoutside'); hold(ax,'off'); drawnow limitrate;
end

% ========================= LOW-LEVEL =========================
function pitch = calcPitchCutY(P)
    W_pack_ext = P.W_Au_top + 2 * P.T_film;
    if P.W > W_pack_ext
        pitch = min(P.W, 0.5 * (P.W + P.W_Au_top + 2 * P.T_film));
    else
        pitch = P.W;
    end
end

function G = buildCellVerts(P, o, isCut, Wy_cell)
    if nargin < 4, Wy_cell = P.W; end

    V_base = vbox(o, P.L, Wy_cell, P.H);
    o_Ti = o + [0 0 P.H];
    V_Ti = vbox(o_Ti, P.L, Wy_cell, P.H_Ti_bottom);
    o_Au = o + [0 0 (P.H + P.H_Ti_bottom)];
    V_Au = vbox(o_Au, P.L, Wy_cell, P.H_gold_bottom);

    z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;

    L_vac_eff = min(max(P.L_vac,0), P.L);
    L_left  = max(0, (P.L - L_vac_eff)/2);
    L_right = max(0, P.L - (L_left + L_vac_eff));
    o_left  = [o(1), o(2), z0_top];
    o_vac   = [o(1) + L_left, o(2), z0_top];
    o_right = [o(1) + L_left + L_vac_eff, o(2), z0_top];
    V_left  = vbox(o_left,  L_left,  Wy_cell, P.H_vac);
    V_vac   = vbox(o_vac,   L_vac_eff, Wy_cell, P.H_vac);
    V_right = vbox(o_right, L_right, Wy_cell, P.H_vac);

    x0_cap = o(1) + (P.L - P.L_Au_top)/2;
    y0_cap = o(2) + (isCut * (Wy_cell - P.W_Au_top - P.T_film) + ~isCut * (Wy_cell - P.W_Au_top)/2);
    z0_cap = z0_top + P.H_vac;
    o_cap  = [x0_cap, y0_cap, z0_cap];
    V_cap  = vbox(o_cap, P.L_Au_top, P.W_Au_top, P.H_Au_top);

    o_film_xL = [x0_cap - P.T_film, y0_cap, z0_cap];
    o_film_xR = [x0_cap + P.L_Au_top, y0_cap, z0_cap];
    V_film_xL = vbox(o_film_xL, P.T_film, P.W_Au_top, P.H_Au_top);
    V_film_xR = vbox(o_film_xR, P.T_film, P.W_Au_top, P.H_Au_top);
    o_film_yF = [x0_cap - P.T_film, y0_cap - P.T_film, z0_cap];
    o_film_yB = [x0_cap - P.T_film, y0_cap + P.W_Au_top, z0_cap];
    V_film_yF = vbox(o_film_yF, P.L_Au_top + 2*P.T_film, P.T_film, P.H_Au_top);
    V_film_yB = vbox(o_film_yB, P.L_Au_top + 2*P.T_film, P.T_film, P.H_Au_top);

    x_in_min = x0_cap - P.T_film;
    x_in_max = x0_cap + P.L_Au_top + P.T_film;
    y_in_min = y0_cap - P.T_film;
    y_in_max = y0_cap + P.W_Au_top + P.T_film;
    x_base_min = o(1); x_base_max = o(1) + P.L;
    y_base_min = o(2); y_base_max = o(2) + Wy_cell;

    T_left_frame  = max(0, x_in_min - x_base_min);
    T_right_frame = max(0, x_base_max - x_in_max);
    T_front_frame = max(0, y_in_min - y_base_min);
    T_back_frame  = max(0, y_base_max - y_in_max);
    z0_frame = z0_cap; H_frame = P.H_Au_top;

    if T_left_frame > 0
        V_frame_xL = vbox([x_base_min, y_base_min, z0_frame], T_left_frame, Wy_cell, H_frame);
    else
        V_frame_xL = [];
    end
    if T_right_frame > 0
        V_frame_xR = vbox([x_in_max, y_base_min, z0_frame], T_right_frame, Wy_cell, H_frame);
    else
        V_frame_xR = [];
    end
    if T_front_frame > 0
        V_frame_yF = vbox([x_in_min, y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame);
    else
        V_frame_yF = [];
    end
    if T_back_frame > 0
        V_frame_yB = vbox([x_in_min, y_in_max, z0_frame], max(0,x_in_max-x_in_min), T_back_frame, H_frame);
    else
        V_frame_yB = [];
    end

    G.base = V_base; G.ti = V_Ti; G.au = V_Au;
    G.left = V_left; G.vac = V_vac; G.right= V_right;
    G.cap = V_cap; G.fxL = V_film_xL; G.fxR = V_film_xR;
    G.fyF = V_film_yF; G.fyB= V_film_yB;
    G.frmL = V_frame_xL; G.frmR = V_frame_xR;
    G.frmF = V_frame_yF; G.frmB = V_frame_yB;
end

function G = buildCellXVerts(P, o, isCut, Lx_cell, Ly_cell)
    if nargin < 5, Ly_cell = P.L; end

    V_base = vbox(o, Lx_cell, Ly_cell, P.H);
    o_Ti = o + [0 0 P.H];
    V_Ti = vbox(o_Ti, Lx_cell, Ly_cell, P.H_Ti_bottom);
    o_Au = o + [0 0 (P.H + P.H_Ti_bottom)];
    V_Au = vbox(o_Au, Lx_cell, Ly_cell, P.H_gold_bottom);

    z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;

    L_vac_eff = min(max(P.L_vac,0), Ly_cell);
    L_left  = max(0, (Ly_cell - L_vac_eff)/2);
    L_right = max(0, Ly_cell - (L_left + L_vac_eff));
    o_left  = [o(1), o(2), z0_top];
    o_vac   = [o(1), o(2) + L_left, z0_top];
    o_right = [o(1), o(2) + L_left + L_vac_eff, z0_top];
    V_left  = vbox(o_left,  Lx_cell, L_left,  P.H_vac);
    V_vac   = vbox(o_vac,   Lx_cell, L_vac_eff, P.H_vac);
    V_right = vbox(o_right, Lx_cell, L_right, P.H_vac);

    x0_cap = o(1) + (isCut * (Lx_cell - P.W_Au_top - P.T_film) + ~isCut * (Lx_cell - P.W_Au_top)/2);
    y0_cap = o(2) + (Ly_cell - P.L_Au_top)/2;
    z0_cap = z0_top + P.H_vac;
    o_cap  = [x0_cap, y0_cap, z0_cap];
    V_cap  = vbox(o_cap, P.W_Au_top, P.L_Au_top, P.H_Au_top);

    o_film_xL = [x0_cap - P.T_film, y0_cap, z0_cap];
    o_film_xR = [x0_cap + P.W_Au_top, y0_cap, z0_cap];
    V_film_xL = vbox(o_film_xL, P.T_film, P.L_Au_top, P.H_Au_top);
    V_film_xR = vbox(o_film_xR, P.T_film, P.L_Au_top, P.H_Au_top);
    o_film_yF = [x0_cap - P.T_film, y0_cap - P.T_film, z0_cap];
    o_film_yB = [x0_cap - P.T_film, y0_cap + P.L_Au_top, z0_cap];
    V_film_yF = vbox(o_film_yF, P.W_Au_top + 2*P.T_film, P.T_film, P.H_Au_top);
    V_film_yB = vbox(o_film_yB, P.W_Au_top + 2*P.T_film, P.T_film, P.H_Au_top);

    x_in_min = x0_cap - P.T_film;
    x_in_max = x0_cap + P.W_Au_top + P.T_film;
    y_in_min = y0_cap - P.T_film;
    y_in_max = y0_cap + P.L_Au_top + P.T_film;
    x_base_min = o(1); x_base_max = o(1) + Lx_cell;
    y_base_min = o(2); y_base_max = o(2) + Ly_cell;

    T_left_frame  = max(0, x_in_min - x_base_min);
    T_right_frame = max(0, x_base_max - x_in_max);
    T_front_frame = max(0, y_in_min - y_base_min);
    T_back_frame = max(0, y_base_max - y_in_max);
    z0_frame = z0_cap; H_frame = P.H_Au_top;

    if T_left_frame > 0
        V_frame_xL = vbox([x_base_min, y_base_min, z0_frame], T_left_frame, Ly_cell, H_frame);
    else
        V_frame_xL = [];
    end
    if T_right_frame > 0
        V_frame_xR = vbox([x_in_max, y_base_min, z0_frame], T_right_frame, Ly_cell, H_frame);
    else
        V_frame_xR = [];
    end
    if T_front_frame > 0
        V_frame_yF = vbox([x_in_min, y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame);
    else
        V_frame_yF = [];
    end
    if T_back_frame > 0
        V_frame_yB = vbox([x_in_min, y_in_max, z0_frame], max(0,x_in_max-x_in_min), T_back_frame, H_frame);
    else
        V_frame_yB = [];
    end

    G.base = V_base; G.ti = V_Ti; G.au = V_Au;
    G.left = V_left; G.vac = V_vac; G.right= V_right;
    G.cap = V_cap; G.fxL = V_film_xL; G.fxR = V_film_xR;
    G.fyF = V_film_yF; G.fyB= V_film_yB;
    G.frmL = V_frame_xL; G.frmR = V_frame_xR;
    G.frmF = V_frame_yF; G.frmB = V_frame_yB;
end

function Gc = buildCentralCrossVacCellVerts(P, o)
    Lside = P.L; Wside = P.L;

    V_base = vbox(o, Lside, Wside, P.H);
    o_Ti = o + [0 0 P.H];
    V_Ti = vbox(o_Ti, Lside, Wside, P.H_Ti_bottom);
    o_Au = o + [0 0 (P.H + P.H_Ti_bottom)];
    V_Au = vbox(o_Au, Lside, Wside, P.H_gold_bottom);

    z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;

    V_mid = vbox([o(1) o(2) z0_top], Lside, Wside, P.H_vac);

    Lx_v = min(P.L_vac, Lside);
    x_v  = o(1) + (Lside - Lx_v)/2;
    V_vacV = vbox([x_v o(2) z0_top], Lx_v, Wside, P.H_vac);

    Wy_h = min(P.L_vac, Lside);
    y_h  = o(2) + (Wside - Wy_h)/2;
    V_vacH = vbox([o(1) y_h z0_top], Lside, Wy_h, P.H_vac);

    capSide = min(P.CapSide_center, Lside);
    x0_cap = o(1) + (Lside - capSide)/2;
    y0_cap = o(2) + (Wside - capSide)/2;
    z0_cap = z0_top + P.H_vac;
    V_cap = vbox([x0_cap, y0_cap, z0_cap], capSide, capSide, P.H_Au_top);

    o_film_xL = [x0_cap - P.T_film, y0_cap, z0_cap];
    o_film_xR = [x0_cap + capSide,  y0_cap, z0_cap];
    V_film_xL = vbox(o_film_xL, P.T_film, capSide, P.H_Au_top);
    V_film_xR = vbox(o_film_xR, P.T_film, capSide, P.H_Au_top);
    o_film_yF = [x0_cap - P.T_film, y0_cap - P.T_film, z0_cap];
    o_film_yB = [x0_cap - P.T_film, y0_cap + capSide,  z0_cap];
    V_film_yF = vbox(o_film_yF, capSide + 2*P.T_film, P.T_film, P.H_Au_top);
    V_film_yB = vbox(o_film_yB, capSide + 2*P.T_film, P.T_film, P.H_Au_top);

    x_in_min = x0_cap - P.T_film;
    x_in_max = x0_cap + capSide + P.T_film;
    y_in_min = y0_cap - P.T_film;
    y_in_max = y0_cap + capSide + P.T_film;
    x_base_min = o(1); x_base_max = o(1) + Lside;
    y_base_min = o(2); y_base_max = o(2) + Wside;

    T_left_frame  = max(0, x_in_min - x_base_min);
    T_right_frame = max(0, x_base_max - x_in_max);
    T_front_frame = max(0, y_in_min - y_base_min);
    T_back_frame  = max(0, y_base_max - y_in_max);
    z0_frame = z0_cap; H_frame = P.H_Au_top;

    if T_left_frame > 0
        V_frame_xL = vbox([x_base_min, y_base_min, z0_frame], T_left_frame, Wside, H_frame);
    else
        V_frame_xL = [];
    end
    if T_right_frame > 0
        V_frame_xR = vbox([x_in_max, y_base_min, z0_frame], T_right_frame, Wside, H_frame);
    else
        V_frame_xR = [];
    end
    if T_front_frame > 0
        V_frame_yF = vbox([x_in_min, y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame);
    else
        V_frame_yF = [];
    end
    if T_back_frame > 0
        V_frame_yB = vbox([x_in_min, y_in_max, z0_frame], max(0,x_in_max-x_in_min), T_back_frame, H_frame);
    else
        V_frame_yB = [];
    end

    Gc.base = V_base; Gc.ti = V_Ti; Gc.au = V_Au;
    Gc.mid = V_mid; Gc.vacH = V_vacH; Gc.vacV = V_vacV;
    Gc.cap = V_cap; Gc.fxL = V_film_xL; Gc.fxR = V_film_xR;
    Gc.fyF = V_film_yF; Gc.fyB= V_film_yB;
    Gc.frmL = V_frame_xL; Gc.frmR = V_frame_xR;
    Gc.frmF = V_frame_yF; Gc.frmB = V_frame_yB;
end

function F = faces6()
    F = [1 2 5 3; 3 5 8 7; 1 3 7 4; 2 6 8 5; 1 4 6 2; 4 7 8 6];
end

function V = vbox(o,Lx,Wy,Hz)
    V = [ o;
          o + [Lx 0 0];
          o + [0 Wy 0];
          o + [0 0 Hz];
          o + [Lx Wy 0];
          o + [Lx 0 Hz];
          o + [0 Wy Hz];
          o + [Lx Wy Hz] ];
end

function [c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeColor,lineWidth] = colorsAndStyle()
    c_base = [0.55 0.35 0.20]; % SiO2
    c_ti   = [0.20 0.20 0.20]; % Ti
    c_au   = [1.00 0.84 0.00]; % Au
    c_vac  = [0.00 0.60 0.00]; % vacuum (visual aid)
    a_vac  = 0.35;
    c_hfo2 = [0.80 0.65 0.45]; % HfO2
    edgeColor = 'k'; lineWidth = 1.0;
end

function drawGeom(ax,G, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, addLegend)
    drawPatch(ax,G.base, c_base, edgeCol,lw, addLegend,'SiO2', 1, F);
    drawPatch(ax,G.ti,   c_ti,   edgeCol,lw, addLegend,'Ti',   1, F);
    drawPatch(ax,G.au,   c_au,   edgeCol,lw, addLegend,'Au',   1, F);

    if isfield(G,'mid') && ~isempty(G.mid)
        drawPatch(ax,G.mid, c_hfo2, edgeCol,lw, addLegend,'HfO2', 1, F);
    else
        if isfield(G,'left') && ~isempty(G.left)
            drawPatch(ax,G.left,  c_hfo2, edgeCol,lw, addLegend,'HfO2', 1, F);
            drawPatch(ax,G.right, c_hfo2, edgeCol,lw, false,'', 1, F);
        end
    end

    if isfield(G,'vacH') && ~isempty(G.vacH)
        drawPatch(ax,G.vacH, c_vac, edgeCol,lw, addLegend,'vacuum', a_vac, F);
        drawPatch(ax,G.vacV, c_vac, edgeCol,lw, false,'', a_vac, F);
    else
        if isfield(G,'vac') && ~isempty(G.vac)
            drawPatch(ax,G.vac,  c_vac, edgeCol,lw, addLegend,'vacuum', a_vac, F);
        end
    end

    drawPatch(ax,G.cap, c_au, edgeCol,lw, false,'', 1, F);
    drawPatch(ax,G.fxL, c_ti, edgeCol,lw, false,'', 1, F);
    drawPatch(ax,G.fxR, c_ti, edgeCol,lw, false,'', 1, F);
    drawPatch(ax,G.fyF, c_ti, edgeCol,lw, false,'', 1, F);
    drawPatch(ax,G.fyB, c_ti, edgeCol,lw, false,'', 1, F);

    drawPatch(ax,G.frmL, c_base, edgeCol,lw, false,'', 1, F);
    drawPatch(ax,G.frmR, c_base, edgeCol,lw, false,'', 1, F);
    drawPatch(ax,G.frmF, c_base, edgeCol,lw, false,'', 1, F);
    drawPatch(ax,G.frmB, c_base, edgeCol,lw, false,'', 1, F);
end

function ph = drawPatch(axh,V,col,edgeCol,lw, addToLegend, name, faceAlpha, F)
    if isempty(V), ph = []; return; end
    if addToLegend && ~isempty(name)
        ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
            'EdgeColor',edgeCol,'LineWidth',lw,'DisplayName',name);
    else
        ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
            'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
    end
end

function FL = buildFillerLayers(o, Lx, Wy, hvec)
    z = o(3);
    FL.V1 = vbox([o(1) o(2) z], Lx, Wy, hvec(1)); z = z + hvec(1);
    FL.V2 = vbox([o(1) o(2) z], Lx, Wy, hvec(2)); z = z + hvec(2);
    FL.V3 = vbox([o(1) o(2) z], Lx, Wy, hvec(3)); z = z + hvec(3);
    FL.V4 = vbox([o(1) o(2) z], Lx, Wy, hvec(4));
end

function drawFillerLayers(ax, FL, c1, c2, c3, c4, edgeCol, lw, F, legendName, offLegend)
    if nargin<10, legendName=''; end
    if nargin<11, offLegend=false; end
    if ~isempty(legendName) && ~offLegend
        patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1,'EdgeColor',edgeCol,'LineWidth',lw, 'DisplayName',legendName);
    else
        patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1,'EdgeColor',edgeCol,'LineWidth',lw, 'HandleVisibility','off');
    end
    patch(ax,'Vertices',FL.V2,'Faces',F,'FaceColor',c2,'EdgeColor',edgeCol,'LineWidth',lw, 'HandleVisibility','off');
    patch(ax,'Vertices',FL.V3,'Faces',F,'FaceColor',c3,'EdgeColor',edgeCol,'LineWidth',lw, 'HandleVisibility','off');
    patch(ax,'Vertices',FL.V4,'Faces',F,'FaceColor',c4,'EdgeColor',edgeCol,'LineWidth',lw, 'HandleVisibility','off');
end

end